﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public class AgreementDuplicateDetection : IPlugin
    {
        private ITracingService tracingService;
        private const string PluginName = "AgreementDuplicateDetection";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];


                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    //This plugin will be triggered by create/updates to Agreement Entity
                    switch (entity.LogicalName)
                    {
                        //handle event first based on initiating entity. 
                        case "ppms_provideragreement":                     
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageCreate:
                                    tracingService.Trace("Create of Provider Agreement: Check for Duplicates");
                                    ProvAgreementCreateCheckForDuplicates(service, entity);
                                    break;
                                case MessageUpdate:
                                    tracingService.Trace("Update of Provider Agreement: Check for Duplicates");
                                    Entity preEntity = (Entity)context.PreEntityImages["PreImage"];   
                                    ProvAgreementUpdateCheckForDuplicates(service, entity, preEntity);
                                    break;                               
                            }
                            break;
                        
                        default:
                            tracingService.Trace("Valid event message/entity not found");
                            break;
                    }
                    
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }     
        private void ProvAgreementCreateCheckForDuplicates(IOrganizationService service, Entity entity)
        {           
            using (var svc = new PpmsContext(service))
            {
                tracingService.Trace("Checking for a Duplicate Agreement of Same Type, Provider, and Status");
                
                //Get the Attributes to check
                OptionSetValue thisAgreementType = entity.GetAttributeValue<OptionSetValue>("ppms_agreementtype");
                var thisProvider = entity.GetAttributeValue<EntityReference>("ppms_account_provideragreementid");
                OptionSetValue thisAgreementStatus = entity.GetAttributeValue<OptionSetValue>("statuscode");

                //Check for an existing Provider Agreement of same type, provider, and status. 
                var matchingProviderAgreement = svc.ppms_provideragreementSet.FirstOrDefault(i => i.ppms_account_ProviderAgreementId.Id == thisProvider.Id && i.ppms_AgreementType == thisAgreementType && i.StatusCode == thisAgreementStatus);
                if (matchingProviderAgreement != null)
                {
                    tracingService.Trace("Duplicate Provider Agreement found");
                    throw new InvalidPluginExecutionException("A Duplicate Provider Agreement of same type, Provider & Status already exists in PPMS. Please check the Provider Agreement list to find existing Provider Agreement.");                  
                }
                tracingService.Trace("No Duplicate Provider Agreement found");
                return;         
            }
        }
        private void ProvAgreementUpdateCheckForDuplicates(IOrganizationService service, Entity entity, Entity preEntity)
        {
            using (var svc = new PpmsContext(service))
            {
                tracingService.Trace("Checking for a Duplicate Agreement of Same Type, Provider, and Status");

                //First check if update contains any of the attributes we are worried about.
                if (entity.Contains("ppms_agreementtype") || entity.Contains("ppms_account_provideragreementid") || entity.Contains("statuscode"))
                {
                    QueryExpression qe = new QueryExpression();
                    qe.EntityName = ppms_provideragreement.EntityLogicalName;
                    qe.ColumnSet.AllColumns = true;

                    var filterExpression = new FilterExpression();
                    filterExpression.FilterOperator = LogicalOperator.And;

                    //Ignore the current Agreement we are updating
                    //var conditionId = new ConditionExpression("ppms_provideragreementid", ConditionOperator.NotEqual, entityId);
                    //filterExpression.AddCondition(conditionId);


                    if (entity.Contains("ppms_agreementtype"))
                    {
                        tracingService.Trace("Checking Agreement Type Update: Entity");
                        OptionSetValue thisAgreementType = entity.GetAttributeValue<OptionSetValue>("ppms_agreementtype");
                        var thisAgreementTypeValue = thisAgreementType.Value;
                        var condition = new ConditionExpression("ppms_agreementtype", ConditionOperator.Equal, (thisAgreementTypeValue.ToString()));
                        filterExpression.AddCondition(condition);
                    }
                    else
                    {
                        //Get the Pre Entity Attribute to check
                        tracingService.Trace("Checking Agreement Type Update: PreEntity");
                        OptionSetValue thisAgreementType = preEntity.GetAttributeValue<OptionSetValue>("ppms_agreementtype");
                        var thisAgreementTypeValue = thisAgreementType.Value;
                        var condition = new ConditionExpression("ppms_agreementtype", ConditionOperator.Equal, (thisAgreementTypeValue.ToString()));
                        filterExpression.AddCondition(condition);
                    }

                    if (entity.Contains("ppms_account_provideragreementid"))
                    {
                        tracingService.Trace("Checking Provider Update: Entity");
                        var thisProvider = entity.GetAttributeValue<EntityReference>("ppms_account_provideragreementid");
                        var condition = new ConditionExpression("ppms_account_provideragreementid", ConditionOperator.Equal, thisProvider.Id);
                        filterExpression.AddCondition(condition);
                    }
                    else
                    {
                        //Get the Pre Entity Attribute to check
                        tracingService.Trace("Checking Provider Update: PreEntity");
                        var thisProvider = preEntity.GetAttributeValue<EntityReference>("ppms_account_provideragreementid");
                        var condition = new ConditionExpression("ppms_account_provideragreementid", ConditionOperator.Equal, thisProvider.Id);
                        filterExpression.AddCondition(condition);
                    }

                    if (entity.Contains("statuscode"))
                    {
                        tracingService.Trace("Checking Status Code Update: Entity");
                        OptionSetValue thisAgreementStatus = entity.GetAttributeValue<OptionSetValue>("statuscode");
                        var thisAgreementStatusValue = thisAgreementStatus.Value;
                        var condition = new ConditionExpression("statuscode", ConditionOperator.Equal, (thisAgreementStatusValue.ToString()));
                        filterExpression.AddCondition(condition);
                    }
                    else
                    {
                        //Get the Pre Entity Attribute to check
                        tracingService.Trace("Checking Status Code Update: PreEntity");
                        OptionSetValue thisAgreementStatus = preEntity.GetAttributeValue<OptionSetValue>("statuscode");
                        var thisAgreementStatusValue = thisAgreementStatus.Value;
                        var condition = new ConditionExpression("statuscode", ConditionOperator.Equal, (thisAgreementStatusValue.ToString()));
                        filterExpression.AddCondition(condition);
                    }

                    qe.Criteria.AddFilter(filterExpression);
                    EntityCollection ec = service.RetrieveMultiple(qe);

                    if (ec.Entities != null)
                    {
                        var provAgreements = ec.Entities.Select(e => e.ToEntity<ppms_provideragreement>()).ToList();
                        //Filter list down to exclude the current Agreemnent being updated. 
                        var matchingAgreements = provAgreements.Where(i => i.Id != entity.Id).ToList();

                        if (matchingAgreements.Any())
                        {
                            tracingService.Trace("Duplicate Provider Agreement found");
                            throw new InvalidPluginExecutionException("A Duplicate Provider Agreement of same type, Provider & Status already exists in PPMS. Please check the Provider Agreement list to find existing Provider Agreement.");
                        }
                    }
                }

                tracingService.Trace("No Duplicate Provider Agreement found");
                return;
            }
        }
    }
}
